unit rVimm;

interface

uses Contnrs, SysUtils, Classes, ORNet, ORFn, ORClasses;

type

  TVimmInputs = record
    noGrid: boolean;
    makeNote: boolean;
    collapseICE: boolean;
    canSaveData: boolean;
    patientName: string;
    patientIEN: string;
    userName: string;
    userIEN: Int64;
    encounterProviderName: string;
    encounterProviderIEN: Int64;
    encounterLocation: integer;
    encounterCategory: string;
    dateEncounterDateTime: TFMDateTime;
    visitString: string;
    documentType: string;
    startInEditMode: boolean;
//    EditList: TStringList;
    DataList: TStringList;
    NewList: TStringList;
  end;

TVimmList = record
  //total list of items for population of additional combo box
  vimmList: TStringList;
  //list of items for population of immunization combobox when documentating an administration at the encounter
  vimmActiveList: TStringList;
  //list of items for population of immunization combobox when documentating historical events
  vimmHistoricalList: TStringList;
  //list of possible routes
  vimmRouteList: TStringList;
  //TStrings of series
  vimmSeries: TStrings;
  //last encounter date
  date: TFmDateTime;
  //first administration code
  firstCode: string;
  //additional administration code
  additionalCodes: string;
end;

//list of vimm data objects to either send to PCE or back to CPRS
TVimmResults = record
  results: TStringList;
end;

//main immunization data object
TVimm = class(TObject)
  ID: string;
  name: string;
  shortName: string;
  maxInSeries: string;
  cvxCode: string;
  inactive: boolean;
  mnemonic: string;
  acronym: string;
  historical: boolean;
  codingSystemList: TStringList;
  visList: TStringList;
  cdcList: TStringList;
  groupList: TStringList;
  synonymList: TStringList;
  lotList: TStringList;
  complete: boolean;
  contraList: TStringList;
  routeDefault: string;
  siteDefault: string;
  doseDefault: string;
  doseUnitDefault: string;
  commentDefault: string;
  constructor Create;
  destructor Destroy; override;
end;

//coding systems used to map CPT and I10 codes to an immunization entry. Can be more than one for an immunization
TVimmCS = class(TObject)
  codingSystem: string;
  code: string;
  narrative: string;
  constructor Create;
  destructor Destroy; override;
end;

//VIS object for an immunization entry. Can be more than one for an immunization.
TVimmVIS = class(TObject)
  id: string;
  name: string;
  effectiveDate: TFMDateTime;
  status: string;
  language: string;
  hl7: string;
  constructor Create;
  destructor Destroy; override;
end;

//Lot object for an immunization. Site defined and can be more than one for an immunization
TVimmLot  = class(TObject)
  id: string;
  name: string;
  manufactureId: string;
  manufactureName: string;
  expirationDate: TFMDateTime;
  ndc: string;
  constructor Create;
  destructor Destroy; override;
end;

//route seperate from immunization. Each route can have a list of site or a flag to determine how the site pick list should work
TVimmRoute = class(TObject)
  text: string;
  all:  boolean;
  none: boolean;
  siteList: TStringList;
  procedure AfterConstruction; override;
  destructor Destroy; override;
end;


//Result object populated from user selection in the UI
TVimmResult = class(TObject)
  id            : string;
  name          : string;
  routeIEN      : string;
  route         : string;
  siteIEN       : string;
  site          : string;
  documType     : string;
  adminByIEN    : string;
  adminBy       : string;
  orderByIEN    : string;
  orderBy       : string;
  visList       : TStringList;
  lotIEN        : string;
  lot           : string;
  seriesId      : string;
  series        : string;
  dosage        : string;
  comments      : string;
  adminDate     : TFMDateTime;
  expirationDate : string;
  manufacturer  : string;
  manufacturerIEN : string;
  outsideLocIEN  : string;
  outsideLoc    : string;
  infoSourceIEN : string;
  infoSource    : string;
  refuseIEN     : string;
  refuse        : string;
  contraIEN     : string;
  contra        : string;
  warnDate      : TFMDateTime;
  procedureCode : TVimmCS;
  diagnosisCode : TVimmCS;
  overrideReason : string;
  noteText      : Tstrings;
  constructor Create; overload;
  constructor Create(input, str2, str3: string); overload;
  destructor Destroy; override;
  function isComplete: boolean;
  function getNoteText: Tstrings;
  function DelimitedStr(uNxtCommSeqNum: integer): string;
  function DelimitedStr2(uNxtCommSeqNum: integer): string;
  function DelimitedStr3(uNxtCommSeqNum: integer): string;
  function procedureDelimitedStr: string;
  function diagnosisDelimitedStr: string;
  procedure setfromDelimitedStr(input, encType: string);
end;

//populate initial comboBox
procedure getIMMShortList(adminDate: TFMDateTime);
procedure setIMMShortList(shortList: TStringList);
procedure setImmShortActiveLookup(immData: TVimm);
procedure setImmShortHistoricalLookup(immData: TVimm);
procedure getShortActiveLookup(var immList: TStringList);
procedure getShortHistoricalLookup(var immList: TStringList);
procedure getBillingCodes(encounterDate: TFmDateTime);
procedure setBillingCodes(encounterDate: TFmDateTime; list: TStringList);
procedure loadManufacturerList(list: TStrings);

//set vimm initial objects and look ups
procedure getImmData(immunization: string; date: TFMDateTime);
procedure setImmDataDetails(tmpList: TStringList);
procedure setTopImmData(var immData: TVimm; tmp: string);
procedure setSubImmData(var immData: TVimm; tmp: string);
procedure getImmRoute(var list: TStringList; date: TFMDateTime);
procedure setImmRoute(tmpList: TStringList);
procedure getImmSite(date: TFMDateTime);
procedure setImmSite(tmpList: TStringList);
function getVImmIds(immunization: string; isHistorical: boolean): string;
procedure getInfoSource(var tempList: TStringList);
procedure getDefaultSite(inputs: TStringList; defaults: TStrings);
procedure getAllSites(list: TStrings);
procedure getRefuseList(date: TFMDateTime; which: string; list: TStrings);
procedure getContraList(immunization: string; list: TStrings);
procedure LoadImmSeries(Dest: TStrings);
function SubSetOfPersons(aReturn: TStrings; const StartFrom: string; Direction: Integer): integer;
function SubSetOfProviders(aReturn: TStrings; const StartFrom: string; Direction: Integer): integer;
procedure LoadHistLocationsVIMM(aReturn: TStrings);
function GetDefLocationsVIMM: TStrings;
procedure getReminders(var AReturn: TStrings);
procedure getReminderMaint(IEN: integer; var aReturn: TStrings);

//procedure and function calls for UI support
function getVimmData(immunization: string): TVimm;
procedure checkForCompleteVimmRecord(immunization: string; date: TFMDateTime);
procedure getVimmLotForImmunization(immunization: string; var returnList: TStringList);
procedure getVimmVISForImmunization(immunization: string; var returnList: TStringList; var lookup: string);
function getMaxSeries(immunization: string): string;
procedure getCodeList(immunization: string; var codeList: TStringList);
procedure getDefaultCodes(var vimmResult: TVimmResult);
procedure getCode(immID, codingSystem, code: string; var vimmResult: TVimmResult);
function getVimmResult(idx: integer): TVimmResult;
function setVimmResults(vimmData: TVimmResult): integer;
function setInitialVimmResult(item, documentType: string): integer;
procedure getVimmResultList(var resultList: TStringList);
function removeVimmResult(immunization: string): boolean;
function checkForWarning(patient, immunization: string; date: TFMDateTime): string;
function findVimmResultsByDelimitedStr(str1, str2, str3: string): TVimmResult;
//function deleteVimmResult(immunization: string): boolean;
procedure clearResults;
procedure clearLists;
procedure clearInputs;

//the following are used when form is open from CPRS coversheet or another application beside CPRS.
function saveData(encDate: TFMDateTime; encLoc, encType, encProv, patient, user: String; var activeList, historicalList, noteList: TStringList): boolean;
procedure buildCurrentPCEList(currList: TStringList; encDate: TFMDateTime; encLoc, encType, encProvider, patient, vstr: string; var PCEList: TStringList);
procedure saveHistoricalData(histList: TStringList; encProvider, patient: string);
procedure setPCEHeader(var PCEList: TStringList; EncCat, EncDate, EncLoc, EncProv, VisitString, patient: string);
procedure SavePCEData(PCEList: TStringList; ANoteIEN, ALocation: integer);
procedure makeNoteList(visitStr: string; encDate: TFMDateTime; encLoc: string; data: TVimmResult; var list: TStringList);
function saveNoteText(noteList: TStringList; encDate: TFMDateTime; encLoc, encType, vstr, patient, user: string): string;
procedure getNoteText(list: TStringList; var vimmResult: TVimMResult);

var
  uVimmList: TVimmList;
  uVimmResults: TVimmResults;
  uVimmSiteList: TStringList;
  uDefLocs : TStringList;
  uVimmInputs: TVimmInputs;

implementation

const
  UpperCaseLetters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
  LowerCaseLetters = 'abcdefghijklmnopqrstuvwxyz';
  Digits = '0123456789';

//pull initial selection list for the main combo box. The values of the list changes based off of documentation type
procedure getIMMShortList(adminDate: TFMDateTime);
var
tmpList: TStringList;
begin
   if (uVimmList.vimmList <> nil) and (uVimmList.vimmList.Count > 0) then exit;
   uVimmList.vimmList := TStringList.Create;
   uVimmList.vimmActiveList := TStringList.Create;
   uVimmList.vimmHistoricalList := TStringList.Create;
   tmpList := TStringList.Create;
   try
    //call for active records
    CallVistA('PXVIMM IMM SHORT LIST',['B', adminDate],tmpList);
    setIMMShortList(tmpList);
   finally
     FreeAndNil(tmpList);
   end;

end;

procedure setIMMShortList(shortList: TStringList);
var
i, idx, j: integer;
immData: TVimm;
tmp: string;
begin
  j := 0;
   for i := 0 to shortList.Count - 1 do
     begin
        if i < j then continue;
        tmp := shortList.Strings[i];
        if Piece(tmp, U, 1) <> 'IMM' then continue;
        idx := uVimmList.vimmList.IndexOf(Piece(tmp, u, 2));
        if idx = -1 then
          begin
            immData := TVimm.Create;
            setTopImmData(immData, Pieces(tmp,u,2,99));
            j := i;
            inc(j);
            if j < shortlist.Count then
              begin
                while Piece(shortList.Strings[j], U, 1) <> 'IMM' do
                  begin
                    tmp := shortList.Strings[j];
                    setSubImmData(immData, tmp);
                    if j = shortlist.Count -1 then break;
                    inc(j);
                  end;
              end;
            immData.complete := false;
            idx := uVimmList.vimmList.addObject(immData.id, immData);
          end;
        if idx = -1 then continue;
        immData := TVimm(uVimmList.vimmList.Objects[idx]);
        if immData.inactive = false then setImmShortActiveLookup(immData);
        if immData.historical = true then setImmShortHistoricalLookup(immData);
     end;
end;

//build lookup text for the combo box
procedure setImmShortActiveLookup(immData: TVimm);
var
c: Integer;
cdc: string;
begin
  uVimmList.vimmActiveList.add(immData.ID + U + immData.name);
  if immData.mnemonic <> '' then uVimmList.vimmActiveList.add(immData.ID + U + immData.mnemonic + ' <'+ immData.name + '>');
  if (immData.cdcList <> nil) and (immData.cdcList.Count > 0) then
  begin
    for c := 0 to immData.cdcList.Count - 1 do
      begin
        cdc := immData.cdcList.Strings[c];
        uVimmList.vimmActiveList.Add(immData.ID + U + cdc + '<'+ immData.name + '>');
      end;
  end;
end;

//build lookup text for the combo box
procedure setImmShortHistoricalLookup(immData: TVimm);
var
c: Integer;
cdc: string;
begin
  uVimmList.vimmHistoricalList.add(immData.ID + U + immData.name);
  if immData.mnemonic <> '' then uVimmList.vimmHistoricalList.add(immData.ID + U + immData.mnemonic + ' <'+ immData.name + '>');
  if (immData.cdcList <> nil) and (immData.cdcList.Count > 0) then
  begin
    for c := 0 to immData.cdcList.Count - 1 do
      begin
        cdc := immData.cdcList.Strings[c];
        uVimmList.vimmHistoricalList.Add(immData.ID + U + cdc + '<'+ immData.name + '>');
      end;
  end;
end;

//return active lookup list to UI
procedure getShortActiveLookup(var immList: TStringList);
begin
  FastAssign(uVimmList.vimmActiveList, immList);
end;

//return historical lookup list to UI
procedure getShortHistoricalLookup(var immList: TStringList);
begin
  FastAssign(uVimmList.vimmHistoricalList, immList);
end;

procedure getBillingCodes(encounterDate: TFMDateTime);
var
aReturn: TStringList;
begin
  if (uVimmList.date > 0) and
  (Piece(FloatToStr(uVimmList.date), '.',1) = Piece(FloatToStr(encounterDate), '.',1)) then exit
  else
    begin
      aReturn := TStringList.Create;
        try
          CallVistA('PXVIMM ADMIN CODES',[encounterDate],aReturn);
          setBillingCodes(encounterDate, aReturn);
        finally
          FreeAndNil(aReturn);
        end;
    end;
end;

procedure setBillingCodes(encounterDate: TFmDateTime; list: TStringList);
var
i: integer;
code, narrative, str: string;
begin
  if StrToInt(list.Strings[0]) < 1 then exit;
  uVimmList.date := encounterDate;
  for i := 1 to list.Count - 1 do
    begin
      str := list.Strings[i];
      code := Piece(str, U, 2);
      narrative := Piece(str, U, 4);
      if POS('CPT-ADM', str) > 0 then uVimmList.firstCode := 'CPT' + '+' + U + code + U + U + narrative + U + '1'
//          uVimmList.firstCode := str;
      else if POS('CPT-ADD', str) > 0 then
        uVimmList.additionalCodes :=  'CPT' + '+' + U + code + U + U + narrative + U + '1';
    end;
end;

procedure loadManufacturerList(list: TSTrings);
var
  aReturn: TStrings;
  i: integer;
begin
  aReturn := TStringList.Create;
  try
    CallVistA('PXVIMM IMM MAN', ['S:B', '','1'], aReturn);
    for i := 1 to aReturn.Count - 1 do
      begin
        list.Add(Pieces(aReturn.Strings[i], U, 1, 2));
      end;
  finally
    FreeAndNil(aReturn);
  end;
end;

//returns details about an immunization
procedure getImmData(immunization: string; date: TFMDateTime)  ;
var
tmpList: TStringList;
begin
  tmpList := TStringList.Create;
  try
    callVistA('PXVIMM IMM DETAILED', [Piece(immunization, U, 1), date], tmpList);
    setImmDataDetails(tmpList);
  finally
     tmpList.Free;
  end;
end;

procedure setImmDataDetails(tmpList: TStringList);
var
tmp: string;
i,idx,j: integer;
immData: TVimm;
begin
  j := 0;
  idx := -1;
  if uVimmList.vimmList = nil then uVimmList.vimmList := TStringList.Create;
  for i := 0 to tmpList.Count -1 do
    begin
     if i < j then continue;
      tmp := tmpList.Strings[i];
      if Piece(tmp, U, 1) <> 'IMM' then continue;
      idx := uVimmList.vimmList.IndexOf(Piece(tmp, u, 2));
      if idx = -1 then immData := TVimm.Create
      else immData := TVimm(uVimmList.vimmList.Objects[idx]);
      setTopImmData(immData, Pieces(tmp,u,2,99));
      j := i;
      inc(j);
      if j = tmpList.Count then exit;
      while Piece(tmpList.Strings[j], U, 1) <> 'IMM' do
        begin
          tmp := tmpList.Strings[j];
          setSubImmData(immData, tmp);
          if j = tmpList.Count -1 then break;
          inc(j);
        end;
    end;
    immData.complete := true;
    if idx > -1 then uVimmList.vimmList.Objects[idx] := immdata
    else uVimmList.vimmList.addObject(immData.id, immData);
end;

//detail setter probably can be combined with setTopIMMData
procedure setTopImmData(var immData: TVimm; tmp: string);
begin
  immData.ID := Piece(tmp, U, 1);
  immData.name := Piece(tmp,u ,2);
  immData.cvxCode := Piece(tmp,u ,3);
  immData.inactive := Piece(tmp,u ,4) = '0';
  immData.historical := Piece(tmp, u, 5) = 'Y';
  immData.mnemonic := Piece(tmp,u ,6);
  immData.acronym := Piece(tmp,u ,7);
  immData.maxInSeries := Piece(tmp, U, 8);
end;

//detail setters for list of data
procedure setSubImmData(var immData: TVimm; tmp: string);
var
vimmType: string;
vimmCS: TVimmCS;
vimmVIS: TVimmVIS;
vimmLot: TVimmLot;

begin
   vimmType := Piece(tmp, U, 1);
   if vimmType = '' then exit;
   if vimmType = 'VIS' then
    begin
      vimmVIS := TVimmVIS.Create;
      vimmVis.id := Piece(tmp, U, 2);
      vimmVIS.name := Piece(tmp, U, 3);
      vimmVIS.effectiveDate := StrToFloat(Piece(tmp, U, 4));
      vimmVIS.status := Piece(tmp, U, 5);
      vimmVIS.language := Piece(tmp, u, 6);
      vimmVIS.hl7 := Piece(tmp, u, 7);
      immData.visList.AddObject(vimmVIS.name + U + FloatToStr(vimmVIS.effectiveDate) + U + vimmVIS.language, vimmVIS);
    end
    else if vimmType = 'CDC' then immData.cdcList.Add(Piece(tmp, U, 2))
    else if vimmType = 'CS' then
      begin
        vimmCS := TVimmCS.Create;
        vimmCS.codingSystem := Piece(tmp, u, 2);
        vimmCS.code := Piece(tmp, U, 3);
        vimmCS.narrative := Piece(tmp, u, 5);
        immData.codingSystemList.AddObject(vimmCS.codingSystem + U + vimmCS.code, vimmCS);
      end
     else if vimmType = 'GROUP' then immData.groupList.Add(Piece(tmp, U, 2))
     else if vimmType = 'SYNONYM' then immData.synonymList.Add(Piece(tmp, U, 2))
     else if vimmType = 'LOT' then
        begin
          vimmLot := TVimmLot.Create;
          vimmLot.id := Piece(tmp, u, 2);
          vimmLot.name := Piece(tmp, u, 3);
          vimmLot.manufactureName := Piece(tmp, u, 4);
          vimmLot.expirationDate := StrToFloat(Piece(tmp, u, 5));
          vimmLot.ndc := Piece(tmp, u, 8);
          immData.lotList.AddObject(vimmLot.name, vimmLot);
        end
     else if vimmType = 'CONTRA' then immData.contraList.Add(Pieces(tmp,u,2,10))
     else if vimmType = 'DEF' then
      begin
        immData.routeDefault := Piece(tmp, U, 2);
        immData.siteDefault := Piece(tmp, U, 3);
        immData.doseDefault := Piece(tmp, U, 4);
        immData.doseUnitDefault := Piece(tmp, U, 6);
      end
     else if vimmType = 'DEFC' then immData.commentDefault := Piece(tmp, U, 2);
end;

//RPC to retreive the route list
procedure getImmRoute(var list: TStringList; date: TFMDateTime);
var
tmpList: TStringList;
begin
  if (uVimmList.vimmRouteList = nil) or (uVimmList.vimmRouteList.Count = 0) then
    begin
      uVimmList.vimmRouteList := TStringList.Create;
      tmpList := TStringList.Create;
      try
        callVistA('PXVIMM ADMIN ROUTE',['S:A', 1], tmpList);
        setImmRoute(tmpList);
      finally
        tmpList.Free;
      end;
    end;
    FastAssign(uVimmList.vimmRouteList, list);
end;

procedure setImmRoute(tmpList: TStringList);
var
i, j, idx: integer;
tmp, route, site: String;

//(1)=ROUTE^something
//(2)=SITE^something
//(...)
//(8)=route^something
begin
  j := 0;
  for i := 0 to tmpList.Count -1 do
    begin
      if i < j then continue;
      tmp := tmpList.Strings[i];
      if Piece(tmp, U, 2) = '' then continue;
      if Piece(tmp, U, 1) <> 'SITE' then
        begin
          route := tmp;
          idx := uVimmList.vimmRouteList.AddObject(Piece(tmp, u, 2), TVimmRoute.Create);
          TVimmRoute(uVimmList.vimmRouteList.objects[idx]).all := false;
          TVimmRoute(uVimmList.vimmRouteList.objects[idx]).none := false;
          TVimmRoute(uVimmList.vimmRouteList.objects[idx]).text := tmp;
          j := i;
          inc(j);
          if j < tmpList.Count then
            begin
              while Piece(tmpList.Strings[j], U, 1) = 'SITE' do
                begin
                  site := tmpList.Strings[j];
                  if Piece(tmpList.Strings[j] , U, 2) = 'ALL' then
                    begin
                      TVimmRoute(uVimmList.vimmRouteList.objects[idx]).all := true;
                    end
                  else if Piece(tmpList.Strings[j] , U, 2) = 'NONE' then
                    begin
                       TVimmRoute(uVimmList.vimmRouteList.objects[idx]).none := true;
                    end
                  else
                    begin
                      TVimmRoute(uVimmList.vimmRouteList.objects[idx]).siteList.Add(Piece(tmpList.Strings[j], U, 2));
                    end;
                if j = tmpList.count - 1 then break;
                inc(j);
              end;
            end;
        end;
    end;
end;

//RPC to get pick list of body sites
procedure getImmSite(date: TFMDateTime);
var
tmpList: TStringList;
begin

  tmpList := TStringList.Create;
  try
  if (uVimmSiteList <> nil) and (uVimmSiteList.Count > 0) then exit;
  callVistA('PXVIMM ADMIN SITE',['S:A', ''], tmpList);
  setImmSite(tmpList);
  finally
   FreeAndNil(tmpList);
  end;
end;

procedure setImmSite(tmpList: TStringList);
var
i: integer;
begin
  if (uVimmSiteList <> nil) and (uVimmSiteList.Count > 0) then exit;
  uVimmSiteList := TStringList.Create;
  for i := 0 to tmpList.Count -1 do
    begin
      if Piece(tmpList.Strings[i], U, 2) = '' then continue;
      uVimmSiteList.Add(tmpList.Strings[i]);
    end;
end;

//retreived the site from the list if the site matches the default from the route
procedure getDefaultSite(inputs: TStringList; defaults: TStrings);
var
i, s: integer;
iSite, site: string;
begin
  for i := 0 to inputs.Count - 1 do
    begin
      iSite := inputs.Strings[i];
      for s := 0 to uVimmSiteList.Count -1 do
        begin
          site := uVimmSiteList.Strings[s];
          if Piece(site, U, 1) = iSite then
            defaults.Add(site);
        end;
    end;
end;

//return all valid sites for the pick list
procedure getAllSites(list: TStrings);
begin
   FastAssign(uVimmSiteList, list);
end;

//RPC to return contradication reason for the pick list
procedure getRefuseList(date: TFMDateTime; which: string; list: TStrings);
var
tmpList: TStringList;
i: integer;
begin
  tmpList := TStringList.Create;
  try
    CallVistA('PXVIMM ICR LIST', [which, 'S:A'], tmpList);
    for i := 0 to tmpList.Count - 1 do
      begin
        if Piece(tmpList.Strings[i], U, 2) = '' then continue;
        list.Add(tmpList.Strings[i]);
      end;
  finally
     tmpList.Free;
  end;
end;

procedure getContraList(immunization: string; list: TStrings);
var
data: TVimm;
i: integer;
tmp: string;
begin
   data := getVimmData(immunization);
   for i := 0 to data.contraList.Count - 1 do
     begin
       tmp := data.contraList.Strings[i];
       list.Add(tmp);
     end;
end;

//RPC to return the list of series for the pick list
procedure LoadImmSeries(Dest: TStrings);
begin
  if uVimmList.vimmSeries = nil then
    begin
      uVimmList.vimmSeries := TStringList.Create;
      callVistA('ORWPCE GET SET OF CODES',['9000010.11','.04','1'], uVimmList.vimmSeries);
    end;
   FastAssign(uVimmList.vimmSeries, dest);
end;

//RPC to return the list of user for the admin by pick list (Copy from rCore)
function SubSetOfPersons(aReturn: TStrings; const StartFrom: string; Direction: Integer): integer;
{ returns a pointer to a list of persons (for use in a long list box) -  The return value is
  a pointer to RPCBrokerV.Results, so the data must be used BEFORE the next broker call! }
begin
    CallVistA('ORWU NEWPERS', [StartFrom, Direction], aReturn);
    Result := aReturn.Count;
end;

//RPC to return the list of provider for the order by pick list (Copy from rCore)
function SubSetOfProviders(aReturn: TStrings; const StartFrom: string; Direction: Integer): integer;
{ returns a pointer to a list of providers (for use in a long list box) -  The return value is
  a pointer to RPCBrokerV.Results, so the data must be used BEFORE the next broker call! }
begin
  CallVistA('ORWU NEWPERS', [startFrom, direction, 'PROVIDER'], aReturn);
  Result := aReturn.count;
end;

procedure LoadHistLocationsVIMM(aReturn: TStrings);
var
  i, j, tlen: integer;
  tmp: string;
begin
  callVistA('ORQQPX GET HIST LOCATIONS', [], aReturn);
  for i := 0 to (aReturn.Count - 1) do
  begin
    tmp := MixedCase(aReturn[i]);
    j := pos(', ',tmp);
    tlen := length(tmp);
    if(j > 0) and (j < (tlen - 2)) and (pos(tmp[j+2],UpperCaseLetters) > 0) and
      (pos(tmp[j+3],LowerCaseLetters)>0) and ((j = (tlen-3)) or (pos(tmp[j+4],LowerCaseLetters)=0)) then
      tmp[j+3] := UpCase(tmp[j+3]);
    if(tlen > 1) then
    begin
      if(pos(tmp[tlen],Digits) > 0) and (pos(tmp[tlen-1],Digits)=0) then
        insert(' ',tmp, tlen);
    end;
    aReturn[i] := tmp;
  end;
end;

function GetDefLocationsVIMM: TStrings;
begin
  if(not assigned(uDefLocs)) then
    begin
      uDefLocs := TStringList.Create;
      CallVistA('ORQQPX GET DEF LOCATIONS', [], uDefLocs);
    end;
  Result := uDefLocs;
end;

procedure getReminders(var AReturn: TStrings);
begin
  CallVista('PXRMRPCV VIMMREM', [uVimmInputs.patientIEN, uVimmInputs.userIEN, uVimmInputs.encounterLocation], AReturn);
end;

procedure getReminderMaint(IEN: integer; var aReturn: TStrings);
begin
  CallVistA('ORQQPXRM REMINDER DETAIL', [uVimmInputs.patientIEN, IEN], aReturn)
end;

//returns the vimm object id for the default values in the immunization pick list
function getVImmIds(immunization: string; isHistorical: boolean): string;
var
idx: integer;
data: TVimm;
id, name: string;

  function findIndexByName(name: string; list: TStringList): string;
   var
   i: integer;
    begin
      result := '';
       for i := 0 to list.Count - 1 do
        begin
          if Piece(list.Strings[i], U, 2) = name then
            begin
              result := Piece(list.Strings[i], U, 1);
              exit;
            end;
        end;
    end;

begin
  result := '';
  id := Piece(immunization, U, 1);
  name := Piece(immunization, U, 2);
  if id = '' then
    begin
      if not isHistorical then id := findIndexByName(name, uVimmList.vimmActiveList)
      else id := findIndexByName(name, uVimmList.vimmHistoricalList);
    end;
  if id = '' then exit;
  idx := uVimmList.vimmList.IndexOf(id);
  if idx = -1 then exit;
  data := TVimm(uVimmList.vimmList.Objects[idx]);
  result := data.ID + U + data.name;
end;

//RPC to get info source values for the pick list
procedure getInfoSource(var tempList: TStringList);
begin
  CallVistA('PXVIMM INFO SOURCE', ['S.A'], tempList);
end;

//start the vimm result object when adding data to the grid.
//this will mainly be populated when the VIMM functionality is called with a default value
//like from Clinical Reminder Dialogs in CPRS
function setInitialVimmResult(item, documentType: string): integer;
var
id, name: string;
data: TVimmResult;
begin
  result := -1;
  id := Piece(item, U, 1);
  name := Piece(item, u, 2);
  if (id = '') or (name = '') then exit;
  data := TVimmResult.Create;
  data.id := id;
  data.name := name;
  if documentType = '0' then data.documType := 'Administered'
  else if documentType = '1' then data.documType := 'Historical'
  else if documentType = '2' then data.documType := 'Contraindication'
  else if documentType = '3' then data.documType := 'Refused';
  result := setVimmResults(data);
end;

//return the vimm object to populate values in the UI
function getVimmData(immunization: string): TVimm;
var
 immidx: integer;
 id, name: string;
begin
  result := nil;
  if immunization = '' then exit;
  name := Piece(immunization,U, 2);
  id := Piece(immunization, u, 1);
  immidx := uVimmList.vimmList.IndexOf(ID);
  if immidx = -1 then exit;
  result := TVimm(uVimmList.vimmList.Objects[immidx]);
end;

//used by the edit panel to determine if the Vimm Object details are populated
//if not call VistA to get the details
procedure checkForCompleteVimmRecord(immunization: string; date: TFMDateTime);
var
data: TVimm;
begin
   data := getVimmData(immunization);
   if data.complete then exit;
   getImmData(Piece(immunization, U, 1), date);
end;

//returns the lot # for an immunization. Use to build the lot pick list
procedure getVimmLotForImmunization(immunization: string; var returnList: TStringList);
var
 l: integer;
 vimmData: TVimm;
 lotData: TVimmLot;
begin
  vimmData := getVimmData(immunization);
  for l := 0 to vimmData.lotList.Count -1 do
    begin
      lotData := TVimmLot(vimmData.lotList.Objects[l]);
      returnList.Add(lotData.id + U + lotData.name + U + lotData.manufactureId + U + lotData.manufactureName + U +
      FormatFMDateTime('mm/dd/yyyy', lotData.expirationDate) + U + lotData.ndc);
    end;
end;

//return the Max Serie value for an immunization. Use to control the series pick list values
function getMaxSeries(immunization: string): string;
var
vimmData: TVimm;
begin
  vimmData := getVimmData(immunization);
  result := vimmData.maxInSeries;
end;

//return a list of procedure code for an immunization entry
procedure getCodeList(immunization: string; var codeList: TStringList);
var
i: integer;
vimmData: TVimm;
vimmCS: TVimmCs;
begin
  vimmData := getVimmData(immunization);
  for i := 0 to vimmData.codingSystemList.Count -1 do
    begin
      vimmCS := TVimmCS(vimmData.codingSystemList.Objects[i]);
      if vimmCS.codingSystem <> 'CPT' then continue;
      codeList.Add(vimmCS.code + U + vimmCS.narrative + ' (' + vimmCS.code + ')');
    end;
end;

//return the list of VIS value for the VIS pick list. lookup is the what the default value should be checked. Determine
//by the most recent activation date and the English language
procedure getVimmVISForImmunization(immunization: string; var returnList: TStringList; var lookup: string);
var
vis, lang, visI: string;
i: integer;
vimmData: TVimm;
visData: TVimmVis;
lastDate, visDate: TFmDateTime;
begin
  vimmData := getVimmData(immunization);
  if vimmData = nil then exit;
  if vimmData.visList = nil then exit;
  lastDate := 0;
  visDate := 0;
  vis := '';
  visI := '-1';
  lang := 'ENGLISH';
  for i := 0 to vimmData.visList.Count - 1 do
    begin
      visData := TVimmVIS(vimmData.visList.Objects[i]);
      visDate := visData.effectiveDate;
      if returnList.IndexOf(visData.name + ' ' + FormatFMDateTime('mm dd, yyyy', visDate) + ' (' + visData.language + ')') = -1 then
        returnList.Add(visData.id + U + visData.name + ' ' + FormatFMDateTime('mm dd, yyyy', visDate) + ' (' + visData.language + ')');
      if (visDate > lastDate) and (visData.language = lang) then
        begin
          lastDate := visDate;
          vis := visData.name;
          visI := IntToStr(I);
        end
      else
        begin
          if lastDate > 0 then visDate := lastDate;
          continue;
        end;
    end;
  if vis = '' then exit;
  lookup := vis + ' ' + FormatFMDateTime('mm dd, yyyy', visDate) + ' (' + lang + ')';
end;

//populate the default codes per coding systems from the Vimm Object.
procedure getDefaultCodes(var vimmResult: TVimmResult);
var
c, idx: integer;
data: TVimm;
codes: TVimmCS;
begin
  idx := uVimmList.vimmList.IndexOf(vimmResult.ID);
  if idx = -1 then exit;
  data := TVimm(uVimmList.vimmList.Objects[idx]);
  if data.codingSystemList = nil then exit;
  for c := 0 to data.codingSystemList.Count - 1 do
    begin
      codes := TVimmCs(data.codingSystemList.Objects[c]);
      if codes.codingSystem = 'CPT' then
        begin
          if (vimmResult.procedureCode <> nil) and (vimmResult.procedureCode.code <> '') then continue;
          vimmResult.procedureCode := codes;
          exit;
        end
      else if codes.codingSystem = 'I10' then
        begin
          if (vimmResult.diagnosisCode <> nil) and (vimmResult.diagnosisCode.code <> '') then continue;
          vimmResult.diagnosisCode := codes;
          exit;
        end;
    end;
end;

procedure getCode(immID, codingSystem, code: string; var vimmResult: TVimmResult);
var
c, idx: integer;
data: TVimm;
codes: TVimmCS;
begin
  idx := uVimmList.vimmList.IndexOf(immID);
  if idx = -1 then exit;
  data := TVimm(uVimmList.vimmList.Objects[idx]);
  if data.codingSystemList = nil then exit;
  c := data.codingSystemList.IndexOf(codingSystem + U + code);
  if c = -1 then exit;
  codes := TVimmCS(data.codingSystemList.Objects[c]);
  if codingSystem = 'CPT' then vimmResult.procedureCode := codes
  else if codingSystem = 'I10' then vimmResult.diagnosisCode := codes;

end;

{ TVimmResult }
function getVimmResult(idx: integer): TVimmResult;
begin
  result := TVimmResult(uVimmResults.results.Objects[idx]);
end;

//add or update an exisiting result in the result list
function setVimmResults(vimmData: TVimmResult): integer;
begin
  if uVimmResults.results = nil then uVimmResults.results := TStringList.Create;
  result := uVimmResults.results.IndexOf(vimmData.name);
  if result > -1 then uVimmResults.results.Objects[result] := vimmData
  else result := uVimmResults.results.AddObject(vimmData.name, vimmData);
end;

//return list of vimm results to the main form to pass on to the calling application
procedure getVimmResultList(var resultList: TStringList);
var
i: integer;
data: TVimmResult;
begin
  if uVimmResults.results = nil then exit;
  for i := 0 to uVimmResults.results.count - 1 do
    begin
      data := getVimmResult(i);
      resultList.AddObject(data.id + U + data.name + U + '0', data);
    end;
end;

function removeVimmResult(immunization: string): boolean;
var
vid: integer;
data: TVimmResult;
begin
  result := false;
  vid := StrToIntDef(Piece(immunization, U, 1), -1);
  if vid < 0 then exit;
  data := getVimmResult(vid);
  uVimmResults.results.Delete(vid);
  data.Destroy;
  result := true;
end;


//RPC to check to see if the paitent and immunization has a warning. Called when selecting
//an immunization
function checkForWarning(patient, immunization: string; date: TFMDateTime): string;
var
tempList: TStringList;
i: integer;
begin
  tempList := TStringList.Create;
  result := '';
  try
     CallVistA('PXVIMM VICR EVENTS', [patient, Piece(immunization, U, 1), date, 'W'], tempList);
     if tempList.Strings[0] = '0' then exit;
     for I := 1 to tempList.Count - 1 do
      begin
         result := result + tempList.Strings[i] + CRLF;
      end;
  finally
     tempList.Free;
  end;
end;

function findVimmResultsByDelimitedStr(str1, str2, str3: string): TVimmResult;
var
i: integer;
data: TVimmResult;
  begin
    result := nil;
    if uVimmResults.results <> nil then
      begin
        for i := 0 to uVimmResults.results.Count - 1 do
          begin
            data := TVimmResult(uVimmResults.results.Objects[i]);
            if data.DelimitedStr(0) <> str1 then continue;
            if data.DelimitedStr2(0) <> str2 then continue;
            if data.DelimitedStr3(0) <> str3 then continue;
            result := data;
            exit
          end;
      end
    else
      begin
        data := TVimmResult.Create(str1, str2, str3);
        result := data;
//        if uVimmResults.results = nil then uVimmResults.results := TStringList.Create;
//        uVimmResults.results.AddObject(data.name, data);
      end;
  end;

//clear result lists;
procedure clearResults;
var
i,input: integer;
data: TVimmResult;
begin
  if uVimmResults.results = nil then exit;
  for i := 0 to uVimmResults.results.count - 1 do
    begin
      data := getVimmResult(i);
      if data <> nil then
        begin
          if uVimmInputs.DataList <> nil then
            begin
              input := uVimmInputs.DataList.IndexOfObject(data);
              if input > -1 then uVimmInputs.DataList.Delete(input);
            end;
          FreeAndNil(data);
        end;
    end;
    uVimmResults.results.Clear;
    FreeAndNil(uVimmResults.results);
end;

//clear lookup list;
procedure clearLists;
var
  i: integer;
  data: TVimm;
  route: TVimmRoute;
begin
  if assigned(uVimmList.vimmActiveList) then
    uVimmList.vimmActiveList.Clear;
  if assigned(uVimmList.vimmHistoricalList) then
    uVimmList.vimmHistoricalList.Clear;

  if assigned(uVimmList.vimmRouteList) then
    begin
      for i := 0 to uVimmList.vimmRouteList.count - 1 do
        begin
          route := TVimmRoute(uVimmList.vimmRouteList.Objects[i]);
          route.Free;
        end;
      FreeAndNil(uVimmList.vimmRouteList);
    end;
  if assigned(uVimmList.vimmSeries) then
    begin
      uVimmList.vimmSeries.Clear;
      FreeAndNil(uVimmList.vimmSeries);
    end;
  if assigned(uVimmList.vimmList) then
  begin
    for i := 0 to uVimmList.vimmList.Count - 1 do
    begin
      data := TVimm(uVimmList.vimmList.Objects[i]);
      data.Free;
    end;
    uVimmList.vimmList.Clear;
    // uVimmSiteList.Clear;
    FreeAndNil(uVimmList);
  end;
  if assigned(uDefLocs) then
    FreeAndNil(uDefLocs);
  if assigned(uVimmSiteList) then
    FreeAndNil(uVimmSiteList);
end;

procedure clearInputs;
var
data: TVimmResult;
i: integer;
begin
//  if not assigned(uVimmInputs) then exit;

  if assigned(uVimmInputs.DataList) then
    begin
      for i := 0 to uVimmInputs.DataList.Count - 1 do
        begin
          data := TVimmResult(uVimmInputs.DataList.Objects[i]);
          if data <> nil then FreeAndNil(data);
        end;
      uVimmInputs.DataList.Clear;
      FreeAndNil(uVimmInputs.DataList);
    end;
  if assigned(uVimmInputs.NewList) then
    begin
      uVimmInputs.NewList.Clear;
      FreeAndNil(uVimmInputs.NewList);
    end;
end;


//build note text for reminders. May be replace with new code that calls VistA
function TVimmResult.getNoteText: Tstrings;
begin
  result := self.noteText;
end;

//build the sting to send data back to PCE
function TVimmResult.DelimitedStr(uNxtCommSeqNum: integer): string;
var
i, idx, v: integer;
visI, visTmp: string;
visData: TVimmVIS;
vimmData: TVimm;
begin
//  Result := inherited DelimitedStr;
  //Result := 'IMM' + Result + U + Series + U + IntToStr(Provider) + U + Reaction;
  //        'IMM'         ^IEN           ^cat^  Narrative^     Series   ^enc provider^Reaction^Contraindicated
  //  comment seq^dose^route^site^lot^manufacture^expiration^eventDate^ordering provider^VIS^remarks
  //^^Refused^Lot^Route^Site^AdminBy^OrderBy^Dosage^Warn;
  if (self.documType = 'Contraindication') or (self.documType = 'Refused') then
    begin
      Result := 'ICR+';
      if self.contraIEN <> '' then result := Result + U + self.contraIEN + U + U + self.contra
      else result := result + U + self.refuseIEN + U + U + self.refuse;
      result := result + U + self.id;
      if self.warnDate >0 then result := result + U + FloatToStr(self.warnDate)
      else result := result + U;
      result := result + U+U+U+U+U;
      setPiece(result, U, 10, IntToStr(UNxtCommSeqNum));
      exit;
    end;
  Result := 'IMM' + '+' + U + self.id + U + U + self.name + U + seriesId +
  U + U + U + U + U + U + U + U + U + U + U + U + U + U + U + U + U;
  if self.adminByIEN <> '' then SetPiece(result, U, 6, self.adminByIEN);
  setPiece(result, U, 10, IntToStr(UNxtCommSeqNum));
  if self.infoSourceIEN <> '' then SetPiece(result, U, 12, ';' + self.infoSourceIEN);
  if self.dosage <> '' then setPiece(result, U, 13, self.dosage + ';mL');
  if self.routeIEN <> '' then SetPiece(result, U, 14, ';;'+ self.routeIEN);
  if self.siteIEN <> '' then SetPiece(result, U, 15, ';;'+ self.siteIEN);

  if self.lotIEN <> '' then SetPiece(result, U, 16, ';' + self.lotIEN);
  if self.manufacturer <> '' then SetPiece(result, U, 17, self.manufacturer);
  if self.expirationDate <> '' then  setPiece(result, U, 18, self.expirationDate);


//  if self.lotIEN <> '' then
//    begin
//      SetPiece(result, U, 17, self.manufacturer);
//      setPiece(result, U, 18, self.expirationDate);
//  end;

  if self.orderByIEN <> '' then SetPiece(result, U, 20, self.orderByIEN);
  if self.overrideReason <> '' then
    begin
      SetPiece(result, U, 23, '1');
      SetPiece(result, U, 24, IntToStr(uNxtCommSeqNum + 1));
    end;
  if (self.visList = nil) or (self.visList.Count = 0)  then exit;

   visTmp := '';
   idx := uVimmList.vimmList.IndexOf(self.ID);
   if idx = -1 then exit;
   vimmData := TVimm(uVimmList.vimmList.Objects[idx]);
   for i := 0 to self.visList.Count - 1 do
     begin
      visI := self.visList.Strings[i];
      for v := 0 to vimmData.visList.Count - 1 do
        begin
          visData := TVimmVIS(vimmData.visList.Objects[v]);
          if visData.id = visI then
            begin
              if visTmp <> '' then visTmp := visTmp + ';';
              visTmp := visTmp + visData.id + '/' + FloatToStr(visData.effectiveDate);
            end;

        end;
     end;
   SetPiece(result, U, 21, visTMP);
end;

//build the comment string to send back to PCE
function TVimmResult.DelimitedStr2(uNxtCommSeqNum: integer): string;
begin
  If self.comments = '' then
  begin
    result := 'COM' + U +  IntToStr(UNxtCommSeqNum) + U + '@';
  end
  else
  begin
    Result := 'COM' + U +  IntToStr(UNxtCommSeqNum) + U + self.comments;
  end;
end;

function TVimmResult.DelimitedStr3(uNxtCommSeqNum: integer): string;
begin
  If self.overrideReason = '' then
  begin
    result := 'COM' + U +  IntToStr(UNxtCommSeqNum) + U + '@';
  end
  else
  begin
    Result := 'COM' + U +  IntToStr(UNxtCommSeqNum) + U + self.overrideReason;
  end;
end;

//build procedure code string to send data back to PCE
function TVimmResult.procedureDelimitedStr: string;
begin
   result := '';
   if self.procedureCode = nil then exit;
   result := 'CPT' + '+' + U + self.procedureCode.code + U + U + self.procedureCode.narrative + U + '1' + U + self.orderByIEN;
end;

procedure TVimmResult.setfromDelimitedStr(input, encType: string);
var
temp, visTemp: string;
visTempList: TStringList;
i: integer;
begin

   id := Piece(input, '~', 2);
   name := Piece(input, '~', 4);
   seriesId := Piece(input, '~', 5);
   adminByIEN := Piece(input, '~', 6);
   orderByIEN := Piece(input, '~', 20);
   if (encType = 'A') or (encType = 'I') then documType := 'Administered'
   else documType := 'Historical';
   temp := Piece(input, '~', 12);
   if temp <> '' then infoSourceIEN := Piece(temp, ';', 2);
   temp := Piece(input, '~', 13);
   if temp <> '' then dosage := Piece(temp, ';', 1);
   temp := Piece(input, '~', 14);
   if temp <> '' then routeIEN := Piece(temp, ';', 3);
   temp := Piece(input, '~', 15);
   if temp <> '' then siteIEN := Piece(temp, ';', 3);
   temp := Piece(input, '~', 16);
   if temp <> '' then lotIEN := Piece(temp, ';', 2);
   manufacturer := Piece(input, '~', 17);
   expirationDate := Piece(input, '~', 18);
   overrideReason := Piece(input, '~', 24);
   comments := Piece(input, '~', 10);
   temp := Piece(input, '~', 21);
   if temp <> '' then
    begin
      if visList = nil then visList := TStringList.Create;

      if POS(';', temp)>0 then
        begin
          visTempList := TStringList.Create;
          try
             PiecestoList(temp, ';', visTempList);
             for i := 0 to visTempList.Count - 1 do
              begin
                visTemp := visTempList.Strings[i];
                visList.Add(Piece(visTemp, '/', 1));
              end;
          finally
             FreeAndNil(visTempList);
          end;
        end
      else
        visList.Add(Piece(temp, '/', 1));
    end;
end;

//build diagnosis code string to send data back to PCE
function TVimmResult.diagnosisDelimitedStr: string;
begin
  result := '';
  if self.diagnosisCode = nil then exit;
  result := 'POV' + '+' + U + self.diagnosisCode.code + U + U + self.diagnosisCode.narrative  + U + U + self.orderByIEN;
end;

constructor TVimmResult.Create;
begin
  visList := TStringList.Create;
end;

constructor TVimmResult.Create(input, str2, str3: string);
var
 temp,visTemp: string;
 visTempList: TStringList;
 i: integer;
begin
  ID := Piece(input, U, 2);
  name := Piece(input, U, 4);
  seriesId := Piece(input, U, 5);
  adminByIEN := Piece(input, U, 6);
  orderByIEN := Piece(input, U, 20);
  // if (data.encType = 'A') or (encType = 'I') then documType := 'Administered'
  // else documType := 'Historical';
  temp := Piece(input, U, 12);
  if temp <> '' then
    infoSourceIEN := Piece(temp, U, 2);
  temp := Piece(input, U, 13);
  if temp <> '' then
    dosage := Piece(temp, ';', 1);
  temp := Piece(input, U, 14);
  if temp <> '' then
    routeIEN := Piece(temp, ';', 3);
  temp := Piece(input, U, 15);
  if temp <> '' then
    siteIEN := Piece(temp, ';', 3);
  temp := Piece(input, U, 16);
  if temp <> '' then
    lotIEN := Piece(temp, ';', 2);
  manufacturer := Piece(input, U, 17);
  expirationDate := Piece(input, U, 18);
  overrideReason := Piece(input, U, 24);
  comments := Piece(str2, U, 3);
  temp := Piece(input, U, 21);
  if temp <> '' then
  begin
    if visList = nil then
      visList := TStringList.Create;
    if POS(';', temp) > 0 then
    begin
      visTempList := TStringList.Create;
      try
        PiecestoList(temp, ';', visTempList);
        for i := 0 to visTempList.Count - 1 do
        begin
          visTemp := visTempList.Strings[i];
          visList.add(Piece(visTemp, '/', 1));
        end;
      finally
        FreeAndNil(visTempList);
      end;
    end
    else
      visList.add(Piece(temp, '/', 1));
  end;
end;

destructor TVimmResult.Destroy;
begin
  if visList <> nil then FreeAndNil(visList);
  procedureCode := nil;
  diagnosisCode := nil;
  if noteText <> nil then FreeAndNil(noteText);
  inherited;
end;

//detemine if the required fields has been populated used by the main grid
function TVimmResult.isComplete: boolean;
begin
 result := true;
 if self.documType = '' then
  begin
    result := false;
    exit;
  end;
  if self.name = '' then
    begin
      result := false;
      exit;
    end;
  if self.documType = 'Administered' then
    begin
      if self.routeIEN = '' then result := false
      else if self.siteIEN = '' then result := false
      else if self.orderByIEN = '' then result := false
      else if self.adminByIEN = '' then result := false
      else if self.routeIEN = '' then result := false
      else if self.siteIEN = '' then result := false;
//      else if self.lotIEN = '' then result := false;

    end
  else if self.documType = 'Historical' then
    begin
      if self.adminDate = 0 then result := false
      else if self.infoSourceIEN = '' then result := false;
    end
  else if self.documType = 'Refused' then
    begin
      if self.refuseIEN = '' then result := false;
    end
  else if self.documType = 'Contradicted' then
    begin
      if self.contraIEN = '' then result := false
    end;

end;

//code to create two list an active list and historical list. Can update PCE with both list.
function saveData(encDate: TFMDateTime; encLoc, encType, encProv, patient, user: String; var activeList, historicalList, noteList: TStringList): boolean;
var
vStr: string;
i: integer;
data: TVimmResult;
begin
  //break entries into current encounter and historical encounters
  result := true;
  vStr := encLoc + ';' + FloatToStr(encDate) + ';' + encType;
  for i := 0 to uVimmResults.results.Count -1 do
    begin
       data := TVimmResult(uVimmResults.results.Objects[i]);
       if data.isComplete = false then
        begin
          result := false;
          exit;
        end;
       if data.documType = 'Historical' then historicalList.AddObject(data.id, data)
       else activeList.AddObject(data.id, data);
       //build note text for autosave
       if noteList.Count > 0 then noteList.Add('');
       if data.noteText <> nil then noteList.AddStrings(data.noteText);
    end;
end;

//set standard PCE input headers
procedure setPCEHeader(var PCEList: TStringList; EncCat, EncDate, EncLoc, EncProv, VisitString, patient: string);
begin
  PCEList.Add('HDR^' + '0' + U + U + VisitString);
  PCEList.Add('VST^DT^' + EncDate);
  PCEList.Add('VST^PT^' + patient);
  if EncCat <> 'E' then
    BEGIN
      PCEList.Add('VST^HL^' + EncLoc);
    END
  else PCEList.Add('VST^OL^' + EncLoc);
  PCEList.Add('VST^VC^' + EncCat);
  PCEList.Add('PRV^' + EncProv);
end;

//set the active, refuse, and contraindication array to send to PCE. WIll be for the encounter date sent in from the calling application
procedure buildCurrentPCEList(currList: TStringList; encDate: TFMDateTime; encLoc, encType, encProvider, patient, vstr: string; var PCEList: TStringList);
var
dString1, dstring2, dString3: string;
i, UNxtCommSeqNum: integer;
data: TVimmResult;
haveHeader: boolean;
begin
    getBillingCodes(encDate);
    haveHeader := false;
    UNxtCommSeqNum := 0;
    for i := 0 to currList.Count -1 do
      begin
        data := TVimmResult(currList.Objects[i]);
        if haveHeader = false then
          begin
            if (encType = 'H') or (encType = 'D')  then
              begin
                vstr := encLoc + ';' + FloatToStr(data.adminDate) + ';D';
                encType := 'D';
                encDate := data.adminDate;
              end;
            if (encType = 'E') then
              begin
                if (data.outsideLocIEN <> '') then encLoc := data.outsideLocIEN
                else encLoc := '0';
              end;

            setPCEheader(PCEList, encType, FloatToStr(encDate), encLoc, encProvider, vstr, patient);
            haveHeader := true;
          end;
        inc(UNxtCommSeqNum);
        dstring1 := data.DelimitedStr(UNxtCommSeqNum);
        PCEList.Add(dstring1);
        dString2 := data.DelimitedStr2(UNxtCommSeqNum);
        PCEList.Add(dstring2);
        inc(UNxtCommSeqNum);
        dString3 := data.DelimitedStr3(UNxtCommSeqNum);
        PCEList.Add(dString3);
        if data.procedureCode <> nil then
          begin
            dstring1 := data.procedureDelimitedStr;
            PCEList.Add(dstring1);
          end;
        if data.diagnosisCode <> nil then
          begin
            dstring1 := data.diagnosisDelimitedStr;
            PCEList.Add(dstring1);
          end;
        if i = 0 then PCEList.Add(uVimmList.firstCode +U + data.orderByIEN)
        else PCEList.Add(uVimmList.additionalCodes + U + data.orderByIEN);
    end;
end;

//build historical inputs for historical data. Can call PCE multiple times
procedure saveHistoricalData(histList: TStringList; encProvider, patient: string);
var
FileDate: TFMDateTime;
AVisitStr, dString1, dstring2, loc: string;
i, UNxtCommSeqNum: integer;
data: TVimmResult;
PCEList: TStringList;
haveHeader: boolean;
begin
  PCEList := TStringList.Create;
  try
    loc := '0';
    while histList.Count > 0 do
      begin
        haveHeader := false;
        FileDate := 0;
        UNxtCommSeqNum := 0;
        for i := histlist.Count - 1 downto 0 do
          begin
            data := TVimmResult(histList.Objects[i]);
            if FileDate = 0 then FileDate := data.adminDate
            else if FileDate <> data.adminDate then continue;
            if not haveHeader then
              begin
                if data.outsideLocIEN <> '' then AVisitStr := data.outsideLocIEN + ';' + FloatToStr(FileDate) + ';E'
                else AVisitStr := '0;' + FloatToStr(FileDate) + ';E';
                setPCEheader(PCEList, 'E', FloatToStr(FileDate), Loc, encProvider, AVisitStr, patient);
                haveHeader := true;
              end;
            dstring1 := data.DelimitedStr(UNxtCommSeqNum);
            PCEList.Add(dstring1);
            dString2 := data.DelimitedStr2(UNxtCommSeqNum);
            PCEList.Add(dstring2);
            histList.Delete(i);
          end;
        if pceList.Count > 0 then
          begin
            SavePCEData(PCEList, 0, StrToInt(loc));
            PCEList.Clear;
          end;
      end;
  finally
     PCEList.Free;
  end;
end;

//RPC to send data to PCE
procedure SavePCEData(PCEList: TStringList; ANoteIEN, ALocation: integer);
begin
  callVistA('ORWPCE SAVE', [PCEList, ANoteIEN, ALocation]);
end;

//TODO: needs work
//create a note structure. Only used from the coversheet in CPRS or if the calling applicaiton is not CPRS
procedure makeNoteList(visitStr: string; encDate: TFMDateTime; encLoc: string; data: TVimmResult; var list: TStringList);
var
imm0, imm1, imm2, imm3, visI, visTMP: string;
date: TFMDateTime;
visData: TVimmVis;
vimmData: TVimm;
i,idx, v: integer;
begin
  if data.documType = 'Historical' then  date := data.adminDate
  else date := encDate;
  imm0 := 'IMM' + U + data.name + U + U + FloatToStr(date) + U + FloatToStr(data.warnDate) + U + data.series +
            U + data.refuse + U + data.contra + U + data.orderBy + U + data.adminBy + U + data.documType + u + data.infoSource;
  list.Add(imm0);
  imm1 := 'LOC' + U + encLoc + U + U + U + data.outsideLoc;
  List.Add(imm1);
  imm2 := 'ROUTE' + U + data.route + U + data.site + U + data.dosage + U + data.adminBy;
  list.Add(imm2);
  imm3 := 'LOT' + U + data.lot + U + data.manufacturer + U + data.expirationDate;
  list.Add(imm3);
  if data.comments <> '' then list.add('COM' + U + data.comments);
  if data.overrideReason <> '' then list.Add('OVER^'+DATA.overrideReason);

  if (data.visList = nil) or (data.visList.Count = 0) then exit;

   visTmp := '';
   idx := uVimmList.vimmList.IndexOf(data.ID);
   if idx = -1 then exit;
   vimmData := TVimm(uVimmList.vimmList.Objects[idx]);
   for i := 0 to data.visList.Count - 1 do
     begin
      visI := data.visList.Strings[i];
      for v := 0 to vimmData.visList.Count - 1 do
        begin
           visData := TVimmVis(vimmData.visList.Objects[v]);
           if visData.id <> visI then continue;
           visTmp := visData.name + U + FloatToStr(visData.effectiveDate) + U + visData.language;
           list.add('VIS' + U + visTMP);
        end;
     end;
end;

//RPC to create a note. Only used from the coversheet in CPRS or if the calling applicaiton is not CPRS
function saveNoteText(noteList: TStringList; encDate: TFMDateTime; encLoc, encType, vstr, patient, user: string): string;
var
aReturn: string;
begin
    CallVistA('PXRMRPCV MAKENOTE', [noteList, FloatToStr(encDate), encLoc, encType, vstr, patient, user], aReturn);
    Result := aReturn;
end;

procedure getNoteText(list: TStringList; var vimmResult: TVimMResult);
var
noteText: TStringList;
begin
  noteText := TStringList.create;
  try
    CallVistA('PXRMRPCV GETTEXT', [list], noteText);
    if vimmResult.noteText = nil then vimmResult.noteText := TStringList.Create
    else vimmResult.noteText.Clear;
    FastAssign(noteText, vimmResult.noteText);
  finally
    noteText.free;
  end;

end;

{ TVimm }

constructor TVimm.Create;
begin
  codingSystemList := TStringList.Create;
  visList := TStringList.Create;
  groupList := TStringList.Create;
  synonymList := TStringList.Create;
  lotList := TStringList.Create;
  contraList := TStringList.Create;
  cdcList := TStringList.Create;
end;

destructor TVimm.Destroy;
begin
  FreeAndNil(codingSystemList);
  FreeAndNil(visList);
  FreeAndNil(groupList);
  FreeAndNil(synonymList);
  FreeAndNil(lotList);
  FreeAndNil(contraList);
  FreeAndNil(cdcList);
  inherited;
end;


{ TVimmRoute }

procedure TVimmRoute.AfterConstruction;
begin
  inherited;
  siteList := TStringList.Create;
end;

destructor TVimmRoute.Destroy;
begin
  if assigned(siteList) then FreeAndNil(SiteList);
  inherited;
end;

{ TVimmCS }

constructor TVimmCS.Create;
begin
//   inherited;
end;

destructor TVimmCS.Destroy;
begin

  inherited;
end;

{ TVimmVIS }

constructor TVimmVIS.Create;
begin

end;

destructor TVimmVIS.Destroy;
begin

  inherited;
end;

{ TVimmLot }

constructor TVimmLot.Create;
begin

end;

destructor TVimmLot.Destroy;
begin

  inherited;
end;

initialization

finalization
  clearLists;

end.
